<?php
require_once '../includes/auth.php';
require_once '../includes/csrf.php';

// We'll render everything client-side via the CRUD API.
// Provide only the CSRF token for use in fetch requests.
$csrfToken = csrf_token();
?>
<?php
// Compute application base by stripping the '/admin' segment from the script path so API_CRUD points to the app root.
$scriptName = $_SERVER['SCRIPT_NAME'] ?? '';
$adminPos = strpos($scriptName, '/admin/');
if ($adminPos !== false) {
    $appBase = substr($scriptName, 0, $adminPos);
} else {
    $adminPos = strpos($scriptName, '/admin');
    if ($adminPos !== false) {
        $appBase = substr($scriptName, 0, $adminPos);
    } else {
        // fallback to parent dir
        $appBase = rtrim(dirname($scriptName), '/');
    }
}
// normalize empty to empty string (root)
if ($appBase === '') $appBase = '';
?>
<!DOCTYPE html>
<html>
<head>
    <title>Admin Dashboard</title>
    <meta name="csrf-token" content="<?= htmlspecialchars($csrfToken) ?>">
    <script>const API_CRUD = '<?= $appBase ?>/includes/crud.php';</script>
    <style>
        body { font-family: Arial, sans-serif; margin: 2em; }
        h2 { margin-top: 1.5em; }
        .section { margin-bottom: 2em; }
        table { border-collapse: collapse; width: 100%; margin-bottom: 1em; }
        th, td { border: 1px solid #ccc; padding: 8px; text-align: left; }
        th { background: #f4f4f4; }
        .actions button { margin-right: 8px; }
        .toolbar { margin: 0.5em 0 1em; display:flex; gap:12px; align-items:center; flex-wrap: wrap; }
        .toolbar input[type="text"], .toolbar select { padding: 6px; }
        tr.editing { background: #fff8e1; }
        /* Toasts */
        #toastContainer { position: fixed; top: 16px; right: 16px; z-index: 1000; display: flex; flex-direction: column; gap: 8px; }
        .toast { background: #333; color: #fff; padding: 10px 14px; border-radius: 4px; box-shadow: 0 2px 8px rgba(0,0,0,0.2); opacity: 0.95; }
        .toast.success { background: #2e7d32; }
        .toast.error { background: #c62828; }
        .toast.info { background: #1565c0; }
        .pager { display:flex; gap:8px; align-items:center; }
        .pager .info { color:#555; }
        .pager a, .pager button { padding:6px 10px; border:1px solid #ccc; background:#fafafa; cursor:pointer; text-decoration:none; color:#333; }
        .pager a.disabled, .pager button[disabled] { opacity:.5; cursor:not-allowed; }
        .per-page { display:flex; align-items:center; gap:6px; }
    </style>
</head>
<body>
    <div id="toastContainer"></div>

    <h1>Dashboard</h1>
    <p>
        <a href="login.php">Logout</a>
    </p>

    <div class="section" id="ticketsSection">
        <h2>Tickets</h2>
    <div class="toolbar">
            <button onclick="toggleAddTicket()">Add Ticket</button>
            <label>Filter by status:
                <select id="ticketStatusFilter" onchange="filterTickets()">
                    <option value="">All</option>
                    <option value="open">open</option>
                    <option value="in progress">in progress</option>
                    <option value="done">done</option>
                </select>
            </label>
            <label>Search:
                <input type="text" id="ticketSearch" oninput="filterTickets()" placeholder="Search title/description">
            </label>
            <div class="per-page">
                <span>Per page:</span>
                <select id="ticketsPP" onchange="setTicketsPP(this.value)">
                    <option value="10" <?= $ticketsPerPage===10?'selected':'' ?>>10</option>
                    <option value="25" <?= $ticketsPerPage===25?'selected':'' ?>>25</option>
                    <option value="50" <?= $ticketsPerPage===50?'selected':'' ?>>50</option>
                </select>
            </div>
            <div class="pager">
                <button id="tickets-prev" disabled>Prev</button>
                <span id="tickets-pager-info" class="info">Page 1 / 1 — 0 items</span>
                <button id="tickets-next" disabled>Next</button>
            </div>
        </div>
        <div id="addTicketForm" class="inline-form" style="display:none; border:1px solid #ddd; padding:12px; margin-bottom:12px;">
            <form onsubmit="submitAddTicketForm(event)">
                <label for="newTicketTitle">Title</label>
                <input id="newTicketTitle" type="text" required>
                <label for="newTicketDescription">Description</label>
                <textarea id="newTicketDescription" rows="3" required></textarea>
                <label for="newTicketStatus">Status</label>
                <select id="newTicketStatus" required>
                    <option value="open">open</option>
                    <option value="in progress">in progress</option>
                    <option value="done">done</option>
                </select>
                <label for="newTicketUserId">Client ID</label>
                <input id="newTicketUserId" type="number" min="1" required>
                <div style="margin-top:8px;">
                    <button type="submit">Create</button>
                    <button type="button" onclick="toggleAddTicket()">Cancel</button>
                </div>
            </form>
        </div>
    <div id="tickets-loading" style="display:none; padding:8px; background:#fff8e1; border:1px solid #ffe082; margin-bottom:8px;">Loading...</div>
    <table id="ticketsTable">
            <thead>
            <tr>
                <th>ID</th>
                <th>Title</th>
                <th>Description</th>
                <th>Status</th>
                <th>Created At</th>
                <th>Actions</th>
            </tr>
            </thead>
            <tbody>
            <!-- Populated client-side via JS -->
            </tbody>
        </table>
    </div>

    <div class="section" id="clientsSection">
        <h2>Clients</h2>
    <div class="toolbar">
            <button onclick="toggleAddClient()">Add Client</button>
            <label>Search:
                <input type="text" id="clientSearch" oninput="filterClients" placeholder="Search name">
            </label>
            <label style="margin-left:8px; display:flex; align-items:center; gap:6px;"><input type="checkbox" id="showArchivedClients"> Show archived</label>
            <div class="per-page">
                <span>Per page:</span>
                <select id="clientsPP" onchange="setClientsPP(this.value)">
                    <option value="10" <?= $clientsPerPage===10?'selected':'' ?>>10</option>
                    <option value="25" <?= $clientsPerPage===25?'selected':'' ?>>25</option>
                    <option value="50" <?= $clientsPerPage===50?'selected':'' ?>>50</option>
                </select>
            </div>
            <div class="pager">
                <button id="clients-prev" disabled>Prev</button>
                <span id="clients-pager-info" class="info">Page 1 / 1 — 0 items</span>
                <button id="clients-next" disabled>Next</button>
            </div>
        </div>
        <div id="addClientForm" class="inline-form" style="display:none; border:1px solid #ddd; padding:12px; margin-bottom:12px;">
            <form onsubmit="submitAddClientForm(event)">
                <label for="newClientName">Client Name</label>
                <input id="newClientName" type="text" required>
                <div style="margin-top:8px;">
                    <button type="submit">Create</button>
                    <button type="button" onclick="toggleAddClient()">Cancel</button>
                </div>
            </form>
        </div>
    <div id="clients-loading" style="display:none; padding:8px; background:#e3f2fd; border:1px solid #90caf9; margin-bottom:8px;">Loading...</div>
    <table id="clientsTable">
            <thead>
            <tr>
                <th>ID</th>
                <th>Name</th>
                <th>Created At</th>
                <th>Actions</th>
            </tr>
            </thead>
            <tbody>
            <!-- Populated client-side via JS -->
            </tbody>
        </table>
    </div>

    <script>
    const CSRF_TOKEN = document.querySelector('meta[name="csrf-token"]').getAttribute('content');

    function showToast(message, type) {
        var cont = document.getElementById('toastContainer');
        var t = document.createElement('div');
        t.className = 'toast ' + (type || 'info');
        t.textContent = message;
        cont.appendChild(t);
        setTimeout(function(){ t.remove(); }, 3000);
    }

    // Fetch-based API helper
    // apiFetch supports an optional opts object as last param: { section: 'tickets'|'clients' }
    async function apiFetch(path, method = 'GET', body = null, opts = {}) {
        var init = { method: method, headers: {} };
        if (method !== 'GET') {
            init.headers['Content-Type'] = 'application/json';
            init.headers['X-CSRF-Token'] = CSRF_TOKEN;
        }
        if (body != null) {
            init.body = typeof body === 'string' ? body : JSON.stringify(body);
        }

        // Toggle loading state for sections if requested
        var section = opts.section || null;
        var loadingEl = null;
        var disableElements = [];
        if (section === 'tickets') {
            loadingEl = document.getElementById('tickets-loading');
            disableElements = [document.getElementById('tickets-prev'), document.getElementById('tickets-next'), document.getElementById('ticketsPP'), document.getElementById('ticketStatusFilter'), document.getElementById('ticketSearch')];
        } else if (section === 'clients') {
            loadingEl = document.getElementById('clients-loading');
            disableElements = [document.getElementById('clients-prev'), document.getElementById('clients-next'), document.getElementById('clientsPP'), document.getElementById('clientSearch')];
        }
        try {
            if (loadingEl) loadingEl.style.display = '';
            disableElements.forEach(function(el){ if (el) el.disabled = true; });

            const res = await fetch(path, init);
            const text = await res.text();
            let json = null;
            try { json = text ? JSON.parse(text) : null; } catch (e) { /* not JSON */ }
            if (!res.ok) {
                const msg = (json && json.error) ? json.error : (text || res.statusText);
                showToast('API error: ' + msg, 'error');
                throw new Error('API error: ' + msg);
            }
            return json;
        } catch (err) {
            showToast('Network or server error: ' + err.message, 'error');
            throw err;
        } finally {
            if (loadingEl) loadingEl.style.display = 'none';
            disableElements.forEach(function(el){ if (el) el.disabled = false; });
        }
    }

    // New: API-driven fetch + render for tickets/clients
    function fetchTicketsApi(page, per_page, status, q) {
        var url = API_CRUD + '?entity=tickets&page=' + encodeURIComponent(page) + '&per_page=' + encodeURIComponent(per_page);
        if (status) url += '&status=' + encodeURIComponent(status);
        if (q) url += '&q=' + encodeURIComponent(q);
        apiFetch(url, 'GET', null, { section: 'tickets' }).then(json => renderTicketsFromApi(json)).catch(()=>{});
    }

    function renderTicketsFromApi(json) {
        if (!json || !Array.isArray(json.data)) return;
        var tbody = document.querySelector('#ticketsTable tbody');
        tbody.innerHTML = '';
        json.data.forEach(function(t){
            var tr = document.createElement('tr');
            tr.id = 'ticket-row-' + t.id;
            tr.setAttribute('data-ticket-id', t.id);
            tr.innerHTML = '<td>' + escapeHtml(t.id) + '</td>' +
                '<td><span class="ticket-title">' + escapeHtml(t.title) + '</span></td>' +
                '<td><span class="ticket-description">' + escapeHtml(t.description) + '</span></td>' +
                '<td><span class="ticket-status">' + escapeHtml(t.status) + '</span></td>' +
                '<td><span class="ticket-created">' + escapeHtml(t.created_at || '') + '</span></td>' +
                '<td class="actions">' +
                '<button onclick="markTicket(' + t.id + ', \"open\")">Open</button>' +
                '<button onclick="markTicket(' + t.id + ', \"in progress\")">In Progress</button>' +
                '<button onclick="markTicket(' + t.id + ', \"done\")">Done</button>' +
                '<button class="edit-btn" onclick="beginEditTicket(' + t.id + ')">Edit</button>' +
                '<button class="save-btn" style="display:none" onclick="saveEditTicket(' + t.id + ')">Save</button>' +
                '<button class="cancel-btn" style="display:none" onclick="cancelEditTicket(' + t.id + ')">Cancel</button>' +
                '<button onclick="deleteTicket(' + t.id + ')">Delete</button>' +
                '</td>';
            tbody.appendChild(tr);
        });
        if (json.meta) {
            document.getElementById('tickets-pager-info').textContent = 'Page ' + json.meta.page + ' / ' + json.meta.total_pages + ' — ' + json.meta.total + ' items';
            document.getElementById('tickets-prev').disabled = json.meta.page <= 1;
            document.getElementById('tickets-next').disabled = json.meta.page >= json.meta.total_pages;
        }
    }

    function fetchClientsApi(page, per_page, q) {
        var url = API_CRUD + '?entity=clients&page=' + encodeURIComponent(page) + '&per_page=' + encodeURIComponent(per_page);
        if (q) url += '&q=' + encodeURIComponent(q);
        var show = document.getElementById('showArchivedClients').checked ? '1' : '0';
        url += '&show_archived=' + show;
        apiFetch(url, 'GET', null, { section: 'clients' }).then(json => renderClientsFromApi(json)).catch(()=>{});
    }

    function renderClientsFromApi(json) {
        if (!json || !Array.isArray(json.data)) return;
        var tbody = document.querySelector('#clientsTable tbody');
        tbody.innerHTML = '';
        json.data.forEach(function(c){
            var tr = document.createElement('tr');
            tr.id = 'client-row-' + c.id;
            tr.setAttribute('data-client-id', c.id);
            tr.innerHTML = '<td>' + escapeHtml(c.id) + '</td>' +
                '<td><span class="client-name">' + escapeHtml(c.name) + '</span></td>' +
                '<td>' + escapeHtml(c.created_at || '') + '</td>' +
                '<td class="actions">' +
                '<button class="edit-btn" onclick="beginEditClient(' + c.id + ')">Edit</button>' +
                '<button class="save-btn" style="display:none" onclick="saveEditClient(' + c.id + ')">Save</button>' +
                '<button class="cancel-btn" style="display:none" onclick="cancelEditClient(' + c.id + ')">Cancel</button>' +
                '<button onclick="deleteClient(' + c.id + ')">Delete</button>' +
                '</td>';
            tbody.appendChild(tr);
        });
        if (json.meta) {
            document.getElementById('clients-pager-info').textContent = 'Page ' + json.meta.page + ' / ' + json.meta.total_pages + ' — ' + json.meta.total + ' items';
            document.getElementById('clients-prev').disabled = json.meta.page <= 1;
            document.getElementById('clients-next').disabled = json.meta.page >= json.meta.total_pages;
        }
    }

    // URL builders for pagination (now AJAX-driven)
    function setTicketsPP(pp) {
        fetchTicketsApi(1, pp, document.getElementById('ticketStatusFilter').value, document.getElementById('ticketSearch').value);
    }
    function setClientsPP(pp) {
        fetchClientsApi(1, pp, document.getElementById('clientSearch').value);
    }

    // wire prev/next and search/status (delegated)
    document.addEventListener('DOMContentLoaded', function(){
        var tPrev = document.getElementById('tickets-prev');
        var tNext = document.getElementById('tickets-next');
        var cPrev = document.getElementById('clients-prev');
        var cNext = document.getElementById('clients-next');
        var ticketsPer = document.getElementById('ticketsPP');
        var clientsPer = document.getElementById('clientsPP');
        tPrev && tPrev.addEventListener('click', function(){
            // read page from pager info (simple parse)
            var info = document.getElementById('tickets-pager-info').textContent || '';
            var m = info.match(/Page (\d+)/);
            var page = m ? parseInt(m[1],10) : 1;
            if (page>1) fetchTicketsApi(page-1, parseInt(ticketsPer.value,10), document.getElementById('ticketStatusFilter').value, document.getElementById('ticketSearch').value);
        });
        tNext && tNext.addEventListener('click', function(){
            var info = document.getElementById('tickets-pager-info').textContent || '';
            var m = info.match(/Page (\d+)/);
            var page = m ? parseInt(m[1],10) : 1;
            fetchTicketsApi(page+1, parseInt(ticketsPer.value,10), document.getElementById('ticketStatusFilter').value, document.getElementById('ticketSearch').value);
        });
        cPrev && cPrev.addEventListener('click', function(){
            var info = document.getElementById('clients-pager-info').textContent || '';
            var m = info.match(/Page (\d+)/);
            var page = m ? parseInt(m[1],10) : 1;
            if (page>1) fetchClientsApi(page-1, parseInt(clientsPer.value,10), document.getElementById('clientSearch').value);
        });
        cNext && cNext.addEventListener('click', function(){
            var info = document.getElementById('clients-pager-info').textContent || '';
            var m = info.match(/Page (\d+)/);
            var page = m ? parseInt(m[1],10) : 1;
            fetchClientsApi(page+1, parseInt(clientsPer.value,10), document.getElementById('clientSearch').value);
        });
    // initial load (defaults)
    fetchTicketsApi(1, 10, document.getElementById('ticketStatusFilter').value, document.getElementById('ticketSearch').value);
    fetchClientsApi(1, 10, document.getElementById('clientSearch').value);
    });

    // Tickets: filtering (client-side within current page)
    function filterTickets() {
        var status = (document.getElementById('ticketStatusFilter').value || '').toLowerCase();
        var q = (document.getElementById('ticketSearch').value || '').toLowerCase();
        var rows = document.querySelectorAll('#ticketsTable tbody tr');
        rows.forEach(function(row){
            var title = row.querySelector('.ticket-title').textContent.toLowerCase();
            var desc = row.querySelector('.ticket-description').textContent.toLowerCase();
            var stat = row.querySelector('.ticket-status').textContent.toLowerCase();
            var matchStatus = !status || stat === status;
            var matchText = !q || title.includes(q) || desc.includes(q);
            row.style.display = (matchStatus && matchText) ? '' : 'none';
        });
    }

    // Clients: filtering (client-side within current page)
    function filterClients() {
        var q = (document.getElementById('clientSearch').value || '').toLowerCase();
        var rows = document.querySelectorAll('#clientsTable tbody tr');
        rows.forEach(function(row){
            var name = row.querySelector('.client-name').textContent.toLowerCase();
            row.style.display = !q || name.includes(q) ? '' : 'none';
        });
    }

    // Ticket actions (unchanged)
    function toggleAddTicket() {
        var el = document.getElementById('addTicketForm');
        if (!el) return;
        el.style.display = (el.style.display === 'none' || el.style.display === '') ? 'block' : 'none';
    }

    function submitAddTicketForm(ev) {
        ev.preventDefault();
        var title = document.getElementById('newTicketTitle').value.trim();
        var description = document.getElementById('newTicketDescription').value.trim();
        var status = document.getElementById('newTicketStatus').value;
    var client_id = document.getElementById('newTicketUserId').value.trim();
    if (!title || !description || !status || !client_id) { showToast('Please fill in all fields', 'error'); return; }
    var payload = { entity: 'tickets', title: title, description: description, status: status, client_id: client_id };
    // Optimistically append a temporary row
    var tempId = 'tmp-' + Date.now();
    appendTicketRow({ id: tempId, title: title, description: description, status: status, client_id: client_id, created_at: (new Date()).toISOString().slice(0,19).replace('T',' ') });
    document.getElementById('newTicketTitle').value = '';
    document.getElementById('newTicketDescription').value = '';
    document.getElementById('newTicketStatus').value = 'open';
    document.getElementById('newTicketUserId').value = '';
    toggleAddTicket();
    apiFetch(API_CRUD, 'POST', payload, { section: 'tickets' }).then(json => {
        showToast('Ticket created', 'success');
        // refresh current tickets page to get accurate totals
        var info = document.getElementById('tickets-pager-info').textContent || '';
        var m = info.match(/Page (\d+)/); var page = m ? parseInt(m[1],10) : 1;
        fetchTicketsApi(page, parseInt(document.getElementById('ticketsPP').value,10) || 10, document.getElementById('ticketStatusFilter').value, document.getElementById('ticketSearch').value);
    }).catch(err => {
        showToast('Failed to create ticket: ' + err.message, 'error');
        // remove temporary row
        var tmp = document.getElementById('ticket-row-' + tempId);
        if (tmp) tmp.remove();
    });
    }

    function appendTicketRow(t) {
        var tbody = document.querySelector('#ticketsTable tbody');
        var tr = document.createElement('tr');
        tr.id = 'ticket-row-' + t.id;
        tr.setAttribute('data-ticket-id', t.id);
        tr.innerHTML = '<td>' + escapeHtml(t.id) + '</td>' +
            '<td><span class="ticket-title">' + escapeHtml(t.title) + '</span></td>' +
            '<td><span class="ticket-description">' + escapeHtml(t.description) + '</span></td>' +
            '<td><span class="ticket-status">' + escapeHtml(t.status) + '</span></td>' +
            '<td><span class="ticket-created">' + escapeHtml(t.created_at || '') + '</span></td>' +
            '<td class="actions">' +
            '<button onclick="markTicket(' + t.id + ', \"open\")">Open</button>' +
            '<button onclick="markTicket(' + t.id + ', \"in progress\")">In Progress</button>' +
            '<button onclick="markTicket(' + t.id + ', \"done\")">Done</button>' +
            '<button class="edit-btn" onclick="beginEditTicket(' + t.id + ')">Edit</button>' +
            '<button class="save-btn" style="display:none" onclick="saveEditTicket(' + t.id + ')">Save</button>' +
            '<button class="cancel-btn" style="display:none" onclick="cancelEditTicket(' + t.id + ')">Cancel</button>' +
            '<button onclick="deleteTicket(' + t.id + ')">Delete</button>' +
            '</td>';
        tbody.prepend(tr);
        filterTickets();
    }

    function beginEditTicket(id) {
        var row = document.getElementById('ticket-row-' + id);
        if (!row) return;
        if (row.classList.contains('editing')) return;
        row.classList.add('editing');
        var titleSpan = row.querySelector('.ticket-title');
        var descSpan = row.querySelector('.ticket-description');
        var titleVal = titleSpan.textContent;
        var descVal = descSpan.textContent;
        titleSpan.innerHTML = '<input type="text" class="edit-title" value="' + escapeHtmlAttr(titleVal) + '">';
        descSpan.innerHTML = '<textarea class="edit-description" rows="2">' + escapeHtml(descVal) + '</textarea>';
        row.querySelector('.edit-btn').style.display = 'none';
        row.querySelector('.save-btn').style.display = '';
        row.querySelector('.cancel-btn').style.display = '';
    }

    function saveEditTicket(id) {
        var row = document.getElementById('ticket-row-' + id);
        if (!row) return;
        var titleInput = row.querySelector('.edit-title');
        var descInput = row.querySelector('.edit-description');
        var title = (titleInput && titleInput.value || '').trim();
        var description = (descInput && descInput.value || '').trim();
        if (!title || !description) { showToast('Title and description required', 'error'); return; }
        var payload = { entity: 'tickets', id: id, title: title, description: description };
        // optimistic UI update
        row.querySelector('.ticket-title').textContent = title;
        row.querySelector('.ticket-description').textContent = description;
        row.classList.remove('editing');
        row.querySelector('.edit-btn').style.display = '';
        row.querySelector('.save-btn').style.display = 'none';
        row.querySelector('.cancel-btn').style.display = 'none';
    apiFetch(API_CRUD, 'PUT', payload, { section: 'tickets' }).then(() => {
            showToast('Ticket updated', 'success');
            // refresh current page
            var info = document.getElementById('tickets-pager-info').textContent || '';
            var m = info.match(/Page (\d+)/); var page = m ? parseInt(m[1],10) : 1;
            fetchTicketsApi(page, parseInt(document.getElementById('ticketsPP').value,10) || 10, document.getElementById('ticketStatusFilter').value, document.getElementById('ticketSearch').value);
        }).catch(err => {
            showToast('Error updating ticket: ' + err.message, 'error');
        });
    }

    function cancelEditTicket(id) {
        var row = document.getElementById('ticket-row-' + id);
        if (!row) return;
        var titleInput = row.querySelector('.edit-title');
        var descInput = row.querySelector('.edit-description');
        if (titleInput) {
            var originalTitle = titleInput.getAttribute('value') || '';
            row.querySelector('.ticket-title').textContent = originalTitle;
        }
        if (descInput) {
            var originalDesc = descInput.textContent || '';
            row.querySelector('.ticket-description').textContent = originalDesc;
        }
        row.classList.remove('editing');
        row.querySelector('.edit-btn').style.display = '';
        row.querySelector('.save-btn').style.display = 'none';
        row.querySelector('.cancel-btn').style.display = 'none';
    }

    function markTicket(id, status) {
        var payload = { entity: 'tickets', id: id, status: status };
        // optimistic status update
        var row = document.getElementById('ticket-row-' + id);
        var old = row ? row.querySelector('.ticket-status').textContent : null;
        if (row) row.querySelector('.ticket-status').textContent = status;
    apiFetch(API_CRUD, 'PUT', payload, { section: 'tickets' }).then(() => {
            showToast('Status updated to ' + status, 'success');
            filterTickets();
        }).catch(err => {
            showToast('Error updating status: ' + err.message, 'error');
            if (row && old !== null) row.querySelector('.ticket-status').textContent = old;
        });
    }

    function deleteTicket(id) {
        if (!confirm('Delete this ticket?')) return;
        // optimistic remove
        var row = document.getElementById('ticket-row-' + id);
        if (row) row.remove();
    apiFetch(API_CRUD + '?entity=tickets&id=' + encodeURIComponent(id), 'DELETE', null, { section: 'tickets' }).then(() => {
            showToast('Ticket deleted', 'success');
            // refresh current page for accurate totals
            var info = document.getElementById('tickets-pager-info').textContent || '';
            var m = info.match(/Page (\d+)/); var page = m ? parseInt(m[1],10) : 1;
            fetchTicketsApi(page, parseInt(document.getElementById('ticketsPP').value,10) || 10, document.getElementById('ticketStatusFilter').value, document.getElementById('ticketSearch').value);
        }).catch(err => {
            showToast('Error deleting ticket: ' + err.message, 'error');
            // could re-add row; simpler: refetch
            var info = document.getElementById('tickets-pager-info').textContent || '';
            var m = info.match(/Page (\d+)/); var page = m ? parseInt(m[1],10) : 1;
            fetchTicketsApi(page, parseInt(document.getElementById('ticketsPP').value,10) || 10, document.getElementById('ticketStatusFilter').value, document.getElementById('ticketSearch').value);
        });
    }

    // Client actions
    function toggleAddClient() {
        var el = document.getElementById('addClientForm');
        if (!el) return;
        el.style.display = (el.style.display === 'none' || el.style.display === '') ? 'block' : 'none';
    }

    function submitAddClientForm(ev) {
        ev.preventDefault();
        var name = document.getElementById('newClientName').value.trim();
        if (!name) { showToast('Please enter a client name', 'error'); return; }
        var payload = { entity: 'clients', name: name };
        // optimistic add
        var tempId = 'tmpc-' + Date.now();
        appendClientRow({ id: tempId, name: name, created_at: (new Date()).toISOString().slice(0,19).replace('T',' ') });
        document.getElementById('newClientName').value = '';
        toggleAddClient();
    apiFetch(API_CRUD, 'POST', payload, { section: 'clients' }).then(json => {
            showToast('Client created', 'success');
            var info = document.getElementById('clients-pager-info').textContent || '';
            var m = info.match(/Page (\d+)/); var page = m ? parseInt(m[1],10) : 1;
            fetchClientsApi(page, parseInt(document.getElementById('clientsPP').value,10) || 10, document.getElementById('clientSearch').value);
        }).catch(err => {
            showToast('Error creating client: ' + err.message, 'error');
            var tmp = document.getElementById('client-row-' + tempId);
            if (tmp) tmp.remove();
        });
    }

    function appendClientRow(c) {
        var tbody = document.querySelector('#clientsTable tbody');
        var tr = document.createElement('tr');
        tr.id = 'client-row-' + c.id;
        tr.setAttribute('data-client-id', c.id);
        tr.innerHTML = '<td>' + escapeHtml(c.id) + '</td>' +
            '<td><span class="client-name">' + escapeHtml(c.name) + '</span></td>' +
            '<td>' + escapeHtml(c.created_at || '') + '</td>' +
            '<td class="actions">' +
            '<button class="edit-btn" onclick="beginEditClient(' + c.id + ')">Edit</button>' +
            '<button class="save-btn" style="display:none" onclick="saveEditClient(' + c.id + ')">Save</button>' +
            '<button class="cancel-btn" style="display:none" onclick="cancelEditClient(' + c.id + ')">Cancel</button>' +
            '<button onclick="deleteClient(' + c.id + ')">Delete</button>' +
            '</td>';
        tbody.prepend(tr);
        filterClients();
    }

    function beginEditClient(id) {
        var row = document.getElementById('client-row-' + id);
        if (!row || row.classList.contains('editing')) return;
        row.classList.add('editing');
        var nameSpan = row.querySelector('.client-name');
        var nameVal = nameSpan.textContent;
        nameSpan.innerHTML = '<input type="text" class="edit-client-name" value="' + escapeHtmlAttr(nameVal) + '">';
        row.querySelector('.edit-btn').style.display = 'none';
        row.querySelector('.save-btn').style.display = '';
        row.querySelector('.cancel-btn').style.display = '';
    }

    function saveEditClient(id) {
        var row = document.getElementById('client-row-' + id);
        if (!row) return;
        var input = row.querySelector('.edit-client-name');
        var name = (input && input.value || '').trim();
        if (!name) { showToast('Name required', 'error'); return; }
        var payload = { entity: 'clients', id: id, name: name };
        // optimistic
        row.querySelector('.client-name').textContent = name;
        row.classList.remove('editing');
        row.querySelector('.edit-btn').style.display = '';
        row.querySelector('.save-btn').style.display = 'none';
        row.querySelector('.cancel-btn').style.display = 'none';
    apiFetch(API_CRUD, 'PUT', payload, { section: 'clients' }).then(() => {
            showToast('Client updated', 'success');
            var info = document.getElementById('clients-pager-info').textContent || '';
            var m = info.match(/Page (\d+)/); var page = m ? parseInt(m[1],10) : 1;
            fetchClientsApi(page, parseInt(document.getElementById('clientsPP').value,10) || 10, document.getElementById('clientSearch').value);
        }).catch(err => {
            showToast('Error updating client: ' + err.message, 'error');
        });
    }

    function cancelEditClient(id) {
        var row = document.getElementById('client-row-' + id);
        if (!row) return;
        var input = row.querySelector('.edit-client-name');
        if (input) {
            var original = input.getAttribute('value') || '';
            row.querySelector('.client-name').textContent = original;
        }
        row.classList.remove('editing');
        row.querySelector('.edit-btn').style.display = '';
        row.querySelector('.save-btn').style.display = 'none';
        row.querySelector('.cancel-btn').style.display = 'none';
    }

    function deleteClient(id) {
        if (!confirm('Delete this client?')) return;
        var row = document.getElementById('client-row-' + id);
        if (row) row.remove();
    apiFetch(API_CRUD + '?entity=clients&id=' + encodeURIComponent(id), 'DELETE', null, { section: 'clients' }).then(() => {
            showToast('Client deleted', 'success');
            var info = document.getElementById('clients-pager-info').textContent || '';
            var m = info.match(/Page (\d+)/); var page = m ? parseInt(m[1],10) : 1;
            fetchClientsApi(page, parseInt(document.getElementById('clientsPP').value,10) || 10, document.getElementById('clientSearch').value);
        }).catch(err => {
            showToast('Error deleting client: ' + err.message, 'error');
            var info = document.getElementById('clients-pager-info').textContent || '';
            var m = info.match(/Page (\d+)/); var page = m ? parseInt(m[1],10) : 1;
            fetchClientsApi(page, parseInt(document.getElementById('clientsPP').value,10) || 10, document.getElementById('clientSearch').value);
        });
    }

    // Helpers
    function escapeHtml(text) {
        var div = document.createElement('div');
        div.textContent = text == null ? '' : text;
        return div.innerHTML;
    }
    function escapeHtmlAttr(text) {
        return (text + '').replace(/&/g,'&amp;').replace(/"/g,'&quot;').replace(/</g,'&lt;').replace(/>/g,'&gt;');
    }
    </script>
</body>
</html>
<?php
// PHP helpers to build pagination URLs while preserving other section params when possible
function buildTicketsUrl(int $page, int $pp): string {
    $params = $_GET;
    $params['tickets_page'] = $page;
    $params['tickets_pp'] = $pp;
    // Do not alter clients params
    $qs = http_build_query($params);
    $path = strtok($_SERVER['REQUEST_URI'], '?');
    return $path . '?' . $qs;
}
function buildClientsUrl(int $page, int $pp): string {
    $params = $_GET;
    $params['clients_page'] = $page;
    $params['clients_pp'] = $pp;
    $qs = http_build_query($params);
    $path = strtok($_SERVER['REQUEST_URI'], '?');
    return $path . '?' . $qs;
}
?>
