<?php
// Secure session cookie flags (set before session_start)
$secure = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off');
session_set_cookie_params([
    'lifetime' => 0,
    'path' => '/',
    'domain' => '',
    'secure' => $secure,
    'httponly' => true,
    'samesite' => 'Lax',
]);

session_start();
require_once '../includes/db.php';
require_once '../includes/csrf.php';

$token = isset($_GET['token']) ? trim((string)$_GET['token']) : '';
$error = '';
$success = '';
$invite = null;

function find_invite($pdo, $token) {
    if ($token === '') return null;
    $stmt = $pdo->prepare('SELECT * FROM admin_invites WHERE token = ?');
    $stmt->execute([$token]);
    return $stmt->fetch();
}

function is_invite_valid($invite) {
    if (!$invite) return false;
    if (!empty($invite['redeemed_at'])) return false;
    if (empty($invite['expires_at'])) return false;
    return (strtotime($invite['expires_at']) > time());
}

// On POST, use token from POST body if provided
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Verify CSRF
    $csrfOk = isset($_POST['csrf_token']) ? csrf_verify($_POST['csrf_token']) : false;
    if (!$csrfOk) {
        $error = 'Invalid request (CSRF). Please refresh and try again.';
    }

    $token = isset($_POST['token']) ? trim((string)$_POST['token']) : $token;
    $invite = find_invite($pdo, $token);

    $password = isset($_POST['password']) ? (string)$_POST['password'] : '';
    $confirm  = isset($_POST['confirm']) ? (string)$_POST['confirm'] : '';

    if (!$invite || !is_invite_valid($invite)) {
        $error = 'Invalid or expired invite token.';
    } elseif ($password === '' || $confirm === '') {
        $error = 'Please enter and confirm your password.';
    } elseif ($password !== $confirm) {
        $error = 'Passwords do not match.';
    } elseif (strlen($password) < 8) {
        $error = 'Password must be at least 8 characters.';
    } else {
        // Check if admin already exists
        $stmt = $pdo->prepare('SELECT id FROM admins WHERE email = ?');
        $stmt->execute([$invite['email']]);
        $exists = $stmt->fetch();
        if ($exists) {
            $error = 'An account with this email already exists.';
        } else {
            try {
                $pdo->beginTransaction();
                $hash = password_hash($password, PASSWORD_DEFAULT);
                $stmt = $pdo->prepare('INSERT INTO admins (email, password_hash) VALUES (?, ?)');
                $stmt->execute([$invite['email'], $hash]);
                $adminId = (int)$pdo->lastInsertId();

                // Mark invite as redeemed
                $stmt = $pdo->prepare('UPDATE admin_invites SET redeemed_at = NOW() WHERE id = ?');
                $stmt->execute([$invite['id']]);

                $pdo->commit();

                // Auto-login
                session_regenerate_id(true);
                $_SESSION['admin_logged_in'] = true;
                $_SESSION['admin_id'] = $adminId;
                $_SESSION['admin_email'] = $invite['email'];

                header('Location: dashboard.php');
                exit;
            } catch (Throwable $e) {
                if ($pdo->inTransaction()) $pdo->rollBack();
                $error = 'Unable to complete registration at this time.';
            }
        }
    }
} else {
    // GET: prefetch invite for display
    $invite = find_invite($pdo, $token);
    if ($token && !$invite) {
        $error = 'Invalid invite link.';
    } elseif ($invite && !is_invite_valid($invite)) {
        $error = 'This invite link has expired or was already used.';
    }
}

$emailDisplay = $invite && !empty($invite['email']) ? $invite['email'] : '';
?>
<!DOCTYPE html>
<html>
<head>
    <title>Admin Registration</title>
    <meta charset="utf-8">
</head>
<body>
    <h2>Admin Registration</h2>
    <?php if (!empty($error)): ?>
        <p style="color:red;"><?= htmlspecialchars($error) ?></p>
    <?php endif; ?>
    <?php if (!empty($success)): ?>
        <p style="color:green;"><?= htmlspecialchars($success) ?></p>
    <?php endif; ?>

    <?php if ($invite && is_invite_valid($invite)): ?>
        <form method="post" autocomplete="off">
            <input type="hidden" name="csrf_token" value="<?= htmlspecialchars(csrf_token()) ?>">
            <div>
                <label>Invite token<br>
                <input type="text" name="token" value="<?= htmlspecialchars($token) ?>" required>
                </label>
            </div>
            <div>
                <label>Email</label><br>
                <input type="email" value="<?= htmlspecialchars($emailDisplay) ?>" disabled>
            </div>
            <div>
                <label>Password</label><br>
                <input type="password" name="password" required minlength="8">
            </div>
            <div>
                <label>Confirm Password</label><br>
                <input type="password" name="confirm" required minlength="8">
            </div>
            <div style="margin-top:8px;">
                <button type="submit">Create Account</button>
            </div>
        </form>
    <?php else: ?>
        <p>Please enter your invite token to register:</p>
        <form method="post" autocomplete="off">
            <input type="hidden" name="csrf_token" value="<?= htmlspecialchars(csrf_token()) ?>">
            <label>Invite token<br>
                <input type="text" name="token" value="<?= htmlspecialchars($token) ?>" required>
            </label>
            <div style="margin-top:8px;"><button type="submit">Check token</button></div>
        </form>
    <?php endif; ?>
</body>
</html>
