<?php
require_once '../includes/auth.php';
require_once '../includes/csrf.php';

$client_id = isset($_GET['client_id']) ? (int)$_GET['client_id'] : 0;
if ($client_id <= 0) {
    http_response_code(400);
    echo "Client id required";
    exit;
}

$csrfToken = csrf_token();
?>
<?php
// Compute application base by stripping the '/admin' segment from the script path so API_CRUD points to the app root.
$scriptName = $_SERVER['SCRIPT_NAME'] ?? '';
$adminPos = strpos($scriptName, '/admin/');
if ($adminPos !== false) {
  $appBase = substr($scriptName, 0, $adminPos);
} else {
  $adminPos = strpos($scriptName, '/admin');
  if ($adminPos !== false) {
    $appBase = substr($scriptName, 0, $adminPos);
  } else {
    // fallback to parent dir
    $appBase = rtrim(dirname($scriptName), '/');
  }
}
if ($appBase === '') $appBase = '';
?>
<!DOCTYPE html>
<html>
<head>
  <meta charset="utf-8">
  <title>Manage Templates for Client #<?= htmlspecialchars($client_id) ?></title>
  <meta name="csrf-token" content="<?= htmlspecialchars($csrfToken) ?>">
  <style>
    body{font-family:Arial, sans-serif;margin:18px}
    .toolbar{display:flex;gap:8px;align-items:center;margin-bottom:12px}
    #templates{list-style:none;padding:0;margin:0;max-width:900px}
    #templates li{border:1px solid #ddd;padding:10px;margin-bottom:6px;background:#fff;display:flex;align-items:center;gap:8px}
    .small{font-size:90%;color:#666}
    #editorModal{position:fixed;left:0;top:0;right:0;bottom:0;background:rgba(0,0,0,.4);display:none;align-items:center;justify-content:center}
    #editorModal .panel{background:#fff;padding:16px;border-radius:6px;max-width:800px;width:100%;max-height:80vh;overflow:auto}
    .drag-handle{cursor:grab;padding:4px 8px;background:#f4f4f4;border-radius:4px}
    .completed{opacity:.6;text-decoration:line-through}
  </style>
</head>
<body>
  <h1>Templates — Client #<?= htmlspecialchars($client_id) ?></h1>
  <p><a href="dashboard.php">Back to Dashboard</a></p>

  <div class="toolbar">
    <button id="btnNew">New Template</button>
    <span class="small">Manage templates and nested template items for this client.</span>
  </div>

  <ul id="templates"></ul>

  <div id="editorModal">
    <div class="panel">
      <h3 id="editorTitle">Template Editor</h3>
      <form id="templateForm" onsubmit="return onTemplateSave(event)">
        <input id="tplTitle" placeholder="Title" style="width:60%" required>
        <input id="tplDesc" placeholder="Short description" style="width:35%">
        <div style="margin-top:12px">
          <h4>Items</h4>
          <div class="small">You can add nested items. Drag top-level items to reorder.</div>
          <ul id="tplItems" style="list-style:none;padding:0;margin:8px 0;max-height:40vh;overflow:auto"></ul>
          <div style="margin-top:8px">
            <input id="newItemText" placeholder="New item text" style="width:60%"> <button id="addItemBtn" type="button">Add Item</button>
          </div>
        </div>
        <div style="margin-top:12px;text-align:right">
          <button type="submit">Save Template</button>
          <button type="button" id="closeEditor">Cancel</button>
        </div>
      </form>
    </div>
  </div>

  <!-- Apply Template Modal -->
  <div id="applyModal" class="modal" style="display:none;position:fixed;left:0;top:0;right:0;bottom:0;background:rgba(0,0,0,.4);align-items:center;justify-content:center">
    <div class="panel" role="dialog" aria-modal="true" aria-labelledby="applyTitle" style="max-width:700px">
      <h3 id="applyTitle">Apply Template</h3>
      <div style="display:flex;gap:12px;align-items:flex-start">
        <div style="flex:1">
          <label>Search ticket: <input id="applyTicketQ" placeholder="Search tickets by title or id"></label>
          <div id="applyTicketResults" style="max-height:200px;overflow:auto;border:1px solid #eee;margin-top:8px;padding:8px"></div>
        </div>
        <div style="width:260px">
          <div id="applyPreview" style="border:1px solid #eee;padding:8px;max-height:240px;overflow:auto"></div>
        </div>
      </div>
      <div style="margin-top:12px;text-align:right;display:flex;gap:8px;align-items:center">
        <div id="applyProgress" style="flex:1;display:none">Applying... <span id="applyProgressText"></span></div>
        <button id="applyCancel">Cancel</button>
        <button id="applyConfirm">Apply</button>
      </div>
    </div>
  </div>

  <div id="toastContainer" style="position:fixed;right:16px;top:16px;z-index:9999"></div>

<script>
const API_CRUD = '<?= $appBase ?>/includes/crud.php';
const CSRF = document.querySelector('meta[name="csrf-token"]').getAttribute('content');

function showToast(msg, type){
  const c = document.getElementById('toastContainer');
  const t = document.createElement('div'); t.textContent = msg; t.className = 'toast ' + (type||'info'); t.style.background = type==='error'?'#c62828':'#333'; t.style.color='white'; t.style.padding='8px'; t.style.borderRadius='4px'; t.style.marginTop='6px';
  c.appendChild(t); setTimeout(()=>t.remove(),3000);
}

async function apiFetch(path, method='GET', body=null){
  const init = { method: method, headers: {} };
  if(method !== 'GET'){
    init.headers['Content-Type'] = 'application/json';
    init.headers['X-CSRF-Token'] = CSRF;
  }
  if(body != null) init.body = JSON.stringify(body);
  const res = await fetch(path, init);
  const text = await res.text();
  let json = null; try{ json = text ? JSON.parse(text) : null } catch(e){}
  if(!res.ok){ const msg = (json && json.error) ? json.error : (text || res.statusText); showToast('API error: ' + msg, 'error'); throw new Error(msg); }
  return json;
}

const clientId = <?= json_encode($client_id) ?>;

async function loadTemplates(){
  try{
  const url = API_CRUD + '?entity=todo_templates&client_id=' + encodeURIComponent(clientId);
    const json = await apiFetch(url, 'GET');
    const list = json && Array.isArray(json.data) ? json.data : [];
    renderTemplates(list);
  }catch(e){ console.error(e); }
}

function renderTemplates(list){
  const ul = document.getElementById('templates'); ul.innerHTML = '';
  list.forEach(t => {
    const li = document.createElement('li'); li.dataset.id = t.id;
    // use server field 'name' for template title
    li.innerHTML = '<div style="flex:1"><strong>' + escapeHtml(t.name || 'Untitled') + '</strong><div class="small">' + escapeHtml(t.description || '') + '</div></div>' +
      '<div class="actions"><button class="edit">Edit</button><button class="apply">Apply</button><button class="del">Delete</button></div>';
    li.querySelector('.edit').addEventListener('click', ()=> openEditor(t.id));
  li.querySelector('.apply').addEventListener('click', ()=> applyTemplate(t.id));
    li.querySelector('.del').addEventListener('click', ()=> deleteTemplate(t.id));
    ul.appendChild(li);
  });
}

function escapeHtml(s){ const d = document.createElement('div'); d.textContent = s==null?'':s; return d.innerHTML; }

// Editor state
let currentTemplate = null; // { id, title, description, items: [...] }

async function openEditor(templateId){
  try{
    if(templateId){
  const json = await apiFetch(API_CRUD + '?entity=todo_templates&id=' + encodeURIComponent(templateId), 'GET');
      const tpl = json && json.data && json.data[0] ? json.data[0] : null;
      currentTemplate = tpl || { id: null, name: '', description: '', items: [] };
      // load template items separately (server stores them in todo_template_items)
      try{
  const itemsResp = await apiFetch(API_CRUD + '?entity=todo_template_items&template_id=' + encodeURIComponent(templateId), 'GET');
        const flat = itemsResp && Array.isArray(itemsResp.data) ? itemsResp.data : itemsResp || [];
        currentTemplate.items = buildTreeFromFlat(flat);
      }catch(e){ currentTemplate.items = currentTemplate.items || []; }
      document.getElementById('editorTitle').textContent = 'Edit Template';
    }else{
      currentTemplate = { id: null, name: '', description: '', items: [] };
      document.getElementById('editorTitle').textContent = 'New Template';
    }
    document.getElementById('tplTitle').value = currentTemplate.name || '';
    document.getElementById('tplDesc').value = currentTemplate.description || '';
    renderTplItems();
    document.getElementById('editorModal').style.display = 'flex';
  }catch(e){ showToast('Failed to load template: ' + e.message, 'error'); }
}

function closeEditor(){ document.getElementById('editorModal').style.display = 'none'; currentTemplate = null; }

function renderTplItems(){
  const ul = document.getElementById('tplItems'); ul.innerHTML = '';
  const items = Array.isArray(currentTemplate.items) ? currentTemplate.items : [];
  items.forEach((it, idx) => {
    ul.appendChild(renderTplItem(it, idx, 0));
  });
}

function renderTplItem(item, idx, depth){
  const li = document.createElement('li'); li.dataset.idx = idx; li.style.marginLeft = (depth * 14) + 'px'; li.style.border='1px solid #eee'; li.style.padding='8px'; li.style.marginBottom='6px';
  li.innerHTML = '<div style="display:flex;gap:8px;align-items:center"><div class="drag-handle">☰</div><div style="flex:1"><div class="text">' + escapeHtml(item.text || '') + '</div><div class="small">' + (item.children && item.children.length ? (item.children.length + ' subitems') : '') + '</div></div><div style="display:flex;flex-direction:column;gap:4px"><input class="est" type="number" placeholder="est min" style="width:80px" value="' + (item.estimated_minutes ? escapeHtml(String(item.estimated_minutes)) : '') + '"><div><button class="add-sub">Add Sub</button><button class="edit">Edit</button><button class="del">Delete</button></div></div></div>';
  li.querySelector('.edit').addEventListener('click', ()=> editTplItem(item));
  li.querySelector('.del').addEventListener('click', ()=> { if(!confirm('Delete item?')) return; deleteTplItemAt(idx); });
  li.querySelector('.add-sub').addEventListener('click', ()=> addSubToItemAt(idx));
  // children
  if (item.children && item.children.length){
    const cul = document.createElement('ul'); cul.style.listStyle='none'; cul.style.padding='0'; cul.style.margin='6px 0 0 0';
    item.children.forEach((c, i) => cul.appendChild(renderTplItem(c, i, depth+1)));
    li.appendChild(cul);
  }
  // top-level drag only
  if(depth===0){ li.draggable = true; li.addEventListener('dragstart', tplDragStart); li.addEventListener('dragover', tplDragOver); li.addEventListener('drop', tplDrop); li.addEventListener('dragend', tplDragEnd); }
  return li;
}


function editTplItem(item){
  // inline edit using prompt fallback
  const text = prompt('Item text', item.text || ''); if(text===null) return; item.text = text; const est = prompt('Estimated minutes (optional)', item.estimated_minutes ? String(item.estimated_minutes) : ''); if(est!==null){ item.estimated_minutes = est===''? null: parseInt(est,10); } renderTplItems();
}

function deleteTplItemAt(index){
  // remove top-level by index
  currentTemplate.items.splice(index,1);
  renderTplItems();
}

function addSubToItemAt(index){
  const txt = prompt('Subitem text'); if(!txt) return; const parent = currentTemplate.items[index]; if(!parent.children) parent.children = []; parent.children.push({ text: txt, children: [] }); renderTplItems(); }

// delete any nested item by reference
function deleteTplItemByRef(item){ if(deleteItemByRef(currentTemplate.items, item)){ renderTplItems(); } }

// add new top-level item
document.getElementById('addItemBtn').addEventListener('click', ()=>{ const v = document.getElementById('newItemText').value.trim(); if(!v) return showToast('Enter text','error'); currentTemplate.items.push({ text: v, children: [] }); document.getElementById('newItemText').value=''; renderTplItems(); });

// build a nested tree from flat template_items rows returned by server
function buildTreeFromFlat(items){
  const map = new Map();
  items.forEach(it => {
    map.set(String(it.id), Object.assign({}, it, { children: [] }));
  });
  const roots = [];
  map.forEach(node => {
    const pid = node.parent_id || null;
    if (pid) {
      const p = map.get(String(pid));
      if (p) p.children.push(node);
      else roots.push(node);
    } else {
      roots.push(node);
    }
  });
  function sortRec(arr){ arr.sort((a,b)=> (a.position||0) - (b.position||0)); arr.forEach(c=> sortRec(c.children)); }
  sortRec(roots);
  // normalize to { text, children }
  function normalize(n){ return { text: n.description || n.text || '', children: (n.children||[]).map(normalize) }; }
  return roots.map(normalize);
}

// Template save: persist todo_templates row and sync its todo_template_items
async function onTemplateSave(ev){ ev.preventDefault(); try{
  const payload = { entity: 'todo_templates', client_id: clientId, name: document.getElementById('tplTitle').value.trim(), description: document.getElementById('tplDesc').value.trim() };
  // create or update template row
  let templateId = currentTemplate && currentTemplate.id ? currentTemplate.id : null;
  if (!templateId) {
  const res = await apiFetch(API_CRUD, 'POST', payload);
    templateId = res && res.id ? res.id : null;
    if (!templateId) throw new Error('Missing template id from server');
    } else {
    payload.id = templateId;
    await apiFetch(API_CRUD, 'PUT', payload);
  }

  // sync items: naive approach - delete existing items on server then recreate from currentTemplate.items
  try{
  const existing = await apiFetch(API_CRUD + '?entity=todo_template_items&template_id=' + encodeURIComponent(templateId), 'GET');
    const flat = existing && Array.isArray(existing.data) ? existing.data : existing || [];
    for(const ex of flat){
  if (ex && ex.id) await apiFetch(API_CRUD + '?entity=todo_template_items&id=' + encodeURIComponent(ex.id), 'DELETE');
    }
  }catch(e){ /* ignore */ }

  // recreate items recursively
  async function createItems(nodes, parentId){
    for(const n of nodes){
      const desc = (n.text || '').trim();
      const est = n.estimated_minutes != null ? parseInt(n.estimated_minutes,10) : null;
      if(!desc) continue;
  const resp = await apiFetch(API_CRUD, 'POST', { entity: 'todo_template_items', template_id: templateId, parent_id: parentId || '', description: desc, estimated_minutes: est });
      const newId = resp && resp.id ? resp.id : null;
      if (n.children && n.children.length && newId) await createItems(n.children, newId);
    }
  }
  await createItems(currentTemplate.items || [], null);

  showToast('Saved','success');
  closeEditor();
  await loadTemplates();
}catch(e){ showToast('Save failed: ' + e.message, 'error'); }}

document.getElementById('closeEditor').addEventListener('click', ()=> closeEditor());

document.getElementById('btnNew').addEventListener('click', ()=> openEditor(null));

async function deleteTemplate(id){ if(!confirm('Delete template?')) return; try{ await apiFetch(API_CRUD + '?entity=todo_templates&id=' + encodeURIComponent(id), 'DELETE'); showToast('Deleted','success'); await loadTemplates(); }catch(e){ showToast('Delete failed: ' + e.message, 'error'); }}

async function applyTemplate(id){
  // Open modal-driven apply flow: ticket search, preview, and server-side apply
  const modal = document.getElementById('applyModal');
  const qInput = document.getElementById('applyTicketQ');
  const results = document.getElementById('applyTicketResults');
  const preview = document.getElementById('applyPreview');
  const progress = document.getElementById('applyProgress');
  const progressText = document.getElementById('applyProgressText');
  const btnConfirm = document.getElementById('applyConfirm');
  const btnCancel = document.getElementById('applyCancel');

  modal.style.display = 'flex';
  modal.setAttribute('aria-hidden', 'false');
  modal.dataset.templateId = id;
  results.innerHTML = '';
  qInput.value = '';
  progress.style.display = 'none';
  btnConfirm.disabled = false; btnCancel.disabled = false;

  // load preview (flat items -> count + total estimated minutes + short list)
  preview.innerHTML = 'Loading preview...';
  try{
  const itemsResp = await apiFetch(API_CRUD + '?entity=todo_template_items&template_id=' + encodeURIComponent(id), 'GET');
    const flat = itemsResp && Array.isArray(itemsResp.data) ? itemsResp.data : itemsResp || [];
    const total = flat.reduce((s,it)=> s + (parseInt(it.estimated_minutes,10) || 0), 0);
    preview.innerHTML = '<div><strong>Items:</strong> ' + flat.length + '</div><div><strong>Total estimated minutes:</strong> ' + total + '</div>';
    const ul = document.createElement('ul'); ul.style.paddingLeft='16px'; ul.style.marginTop='8px';
    flat.slice(0,50).forEach(it=>{
      const li = document.createElement('li'); li.textContent = (it.description || it.text || '').replace(/\s+/g,' ').trim() + (it.estimated_minutes?(' ('+it.estimated_minutes+'m)'):'');
      ul.appendChild(li);
    });
    if(flat.length) preview.appendChild(ul);
  }catch(e){ preview.innerHTML = 'Failed to load preview: ' + e.message; }

  // focus management / simple trap
  const focusables = modal.querySelectorAll('button, input, [tabindex]');
  let lastFocused = document.activeElement;
  qInput.focus();
  function onKey(e){
    if(e.key === 'Escape'){ closeApply(); }
    if(e.key === 'Tab'){
      const first = focusables[0], last = focusables[focusables.length-1];
      if(e.shiftKey && document.activeElement === first){ e.preventDefault(); last.focus(); }
      else if(!e.shiftKey && document.activeElement === last){ e.preventDefault(); first.focus(); }
    }
  }
  modal.addEventListener('keydown', onKey);

  // ticket search (debounced)
  let searchTimer = null; let selectedTicket = null;
  qInput.addEventListener('input', onQuery);
  async function onQuery(){
    clearTimeout(searchTimer); searchTimer = setTimeout(async ()=>{
      const q = qInput.value.trim(); if(!q){ results.innerHTML = ''; selectedTicket = null; return; }
      results.innerHTML = 'Searching...';
      try{
  const sr = await apiFetch(API_CRUD + '?entity=tickets&q=' + encodeURIComponent(q), 'GET');
        const tickets = sr && Array.isArray(sr.data) ? sr.data : sr || [];
        renderTicketResults(tickets);
      }catch(e){ results.innerHTML = 'Search failed'; }
    }, 250);
  }

  function renderTicketResults(tickets){
    results.innerHTML = '';
    if(!tickets || !tickets.length){ results.textContent = 'No tickets'; return; }
    tickets.forEach(t=>{
      const el = document.createElement('div'); el.style.padding='6px'; el.style.borderBottom='1px solid #f8f8f8';
      el.textContent = '#' + (t.id||'') + ' — ' + (t.title || t.name || t.subject || '');
      el.tabIndex = 0;
      el.addEventListener('click', ()=>{ selectedTicket = t; Array.from(results.children).forEach(c=>c.style.background=''); el.style.background='#eef'; qInput.value = t.id || ''; });
      el.addEventListener('keydown', (ev)=>{ if(ev.key === 'Enter') el.click(); });
      results.appendChild(el);
    });
  }

  function cleanupListeners(){
    modal.removeEventListener('keydown', onKey);
    qInput.removeEventListener('input', onQuery);
    searchTimer && clearTimeout(searchTimer);
  }

  function closeApply(){
    cleanupListeners();
    modal.style.display = 'none';
    modal.setAttribute('aria-hidden', 'true');
    selectedTicket = null;
    if(lastFocused && lastFocused.focus) lastFocused.focus();
  }

  btnCancel.onclick = ()=>{ closeApply(); };

  btnConfirm.onclick = async ()=>{
    const tId = selectedTicket ? selectedTicket.id : parseInt(qInput.value,10);
    if(!tId || tId <= 0){ showToast('Please select a valid ticket to apply to', 'error'); return; }
    try{
      progress.style.display = 'block'; progressText.textContent = '';
      btnConfirm.disabled = true; btnCancel.disabled = true;
  const resp = await apiFetch(API_CRUD + '?action=apply_template', 'POST', { template_id: id, ticket_id: tId });
      const created = resp && (resp.created || resp.count || resp.created_count) ? (resp.created || resp.count || resp.created_count) : null;
      progressText.textContent = created ? (created + ' items created') : 'Applied';
      showToast('Template applied to ticket #' + tId, 'success');
      // navigate to ticket page
      window.location.href = '/admin/ticket.php?id=' + encodeURIComponent(tId);
    }catch(e){ showToast('Apply failed: ' + e.message, 'error'); btnConfirm.disabled = false; btnCancel.disabled = false; progress.style.display = 'none'; }
  };
}

// Drag handlers for tpl items (top-level)
let tplDragSrc = null;
function tplDragStart(e){ this.style.opacity='0.5'; tplDragSrc = this; e.dataTransfer.effectAllowed = 'move'; }
function tplDragOver(e){ e.preventDefault(); e.dataTransfer.dropEffect = 'move'; const target = this; const rect = target.getBoundingClientRect(); const next = (e.clientY - rect.top) > (rect.height/2); target.parentNode.insertBefore(tplDragSrc, next ? target.nextSibling : target); }
function tplDrop(e){ e.stopPropagation(); }
function tplDragEnd(e){ this.style.opacity=''; // recalc order of top-level items
  const ul = document.getElementById('tplItems'); const ids = Array.from(ul.children).map((li, idx)=>({ idx: idx, text: li.querySelector('.text').textContent })); // rebuild currentTemplate.items order
  const newItems = [];
  Array.from(ul.children).forEach(li => {
    const text = li.querySelector('.text').textContent;
    // find matching item in old list (by text) - safe enough for small lists
    const found = currentTemplate.items.find(it => it.text === text && !it._used);
    if(found){ found._used = true; newItems.push(found); }
  });
  // append unused
  currentTemplate.items.forEach(it=>{ if(!it._used){ newItems.push(it); } delete it._used; });
  currentTemplate.items = newItems;
  renderTplItems();
}

// helper to delete item from nested structure by reference - used by delete buttons on child items
function deleteItemByRef(parentArr, ref){ const idx = parentArr.indexOf(ref); if(idx>=0){ parentArr.splice(idx,1); return true; } for(const ch of parentArr){ if(ch.children && ch.children.length){ if(deleteItemByRef(ch.children, ref)) return true; } } return false; }

// init
loadTemplates();
</script>
</body>
</html>
