#!/usr/bin/env bash
# Produce a release zip for the Checklist app
# Usage: ./scripts/package_release.sh [release-name]
# Default release-name is 'checklist'

set -euo pipefail

REPO_ROOT=$(cd "$(dirname "${BASH_SOURCE[0]}")/.." && pwd)
cd "$REPO_ROOT"

RELEASE_NAME=${1:-checklist}
TIMESTAMP=$(date -u +"%Y%m%dT%H%M%SZ")
OUT_DIR="$REPO_ROOT/releases"
mkdir -p "$OUT_DIR"

ZIP_NAME="$OUT_DIR/${RELEASE_NAME}-${TIMESTAMP}.zip"
SHA_NAME="$ZIP_NAME.sha256"

# Exclude patterns (add more if needed)
EXCLUDES=(".git" "node_modules" "releases" "vendor" "*.log" "*.sqlite" "env" ".env" )
EXCLUDE_ARGS=()
for ex in "${EXCLUDES[@]}"; do
  EXCLUDE_ARGS+=(--exclude="$ex")
done

# Use zip to create an archive from the current directory, excluding patterns
# If zip is not available, fallback to using tar.gz
if command -v zip >/dev/null 2>&1; then
  echo "Creating zip: $ZIP_NAME"
  # zip -r preserves symlinks etc.; using -x to exclude
  # Build exclude patterns for zip: patterns with paths
  ZIP_EXCLUDES=()
  for ex in "${EXCLUDES[@]}"; do
    ZIP_EXCLUDES+=("-x" "$ex/*" "$ex")
  done
  # zip -r needs list of files, so archive everything in current dir
  # Use a subshell to avoid including releases dir when it already exists
  (shopt -s dotglob; zip -r "$ZIP_NAME" . "${ZIP_EXCLUDES[@]}")
else
  echo "zip not found, creating tar.gz instead"
  TAR_NAME="$OUT_DIR/${RELEASE_NAME}-${TIMESTAMP}.tar.gz"
  tar --exclude-vcs --exclude="releases" --exclude="node_modules" --exclude=".git" -czf "$TAR_NAME" .
  echo "Created $TAR_NAME"
  exit 0
fi

# Generate sha256 checksum
if command -v sha256sum >/dev/null 2>&1; then
  sha256sum "$ZIP_NAME" > "$SHA_NAME"
  echo "Checksum written to $SHA_NAME"
elif command -v shasum >/dev/null 2>&1; then
  shasum -a 256 "$ZIP_NAME" > "$SHA_NAME"
  echo "Checksum written to $SHA_NAME"
else
  echo "Warning: No sha256 checksum utility found; skipping checksum"
fi

echo "Created package: $ZIP_NAME"
ls -lh "$ZIP_NAME" || true

echo "Done."