<?php
// Secure session cookie flags (set before session_start)
$secure = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off');
session_set_cookie_params([
    'lifetime' => 0,
    'path' => '/',
    'domain' => '',
    'secure' => $secure,
    'httponly' => true,
    'samesite' => 'Lax',
]);

session_start();
require_once '../includes/db.php';

// Simple throttle settings
const LOGIN_MAX_ATTEMPTS = 5;
const LOGIN_WINDOW_SECONDS = 15 * 60; // 15 minutes
const LOGIN_LOCK_SECONDS = 5 * 60;    // 5 minutes lock after exceeding

function throttle_file_path($ip) {
    $safeIp = preg_replace('/[^a-zA-Z0-9_.-]/', '_', $ip);
    return sys_get_temp_dir() . DIRECTORY_SEPARATOR . 'checklist_login_' . $safeIp . '.json';
}

function get_ip_throttle($ip) {
    $path = throttle_file_path($ip);
    if (!file_exists($path)) return ['attempts' => 0, 'first_at' => 0, 'locked_until' => 0];
    $raw = @file_get_contents($path);
    if ($raw === false) return ['attempts' => 0, 'first_at' => 0, 'locked_until' => 0];
    $data = json_decode($raw, true);
    if (!is_array($data)) return ['attempts' => 0, 'first_at' => 0, 'locked_until' => 0];
    return $data + ['attempts' => 0, 'first_at' => 0, 'locked_until' => 0];
}

function save_ip_throttle($ip, $data) {
    $path = throttle_file_path($ip);
    @file_put_contents($path, json_encode($data));
}

$error = '';
$ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
$ipThrottle = get_ip_throttle($ip);
$now = time();

// Reset window if expired
if ($ipThrottle['first_at'] && ($now - $ipThrottle['first_at'] > LOGIN_WINDOW_SECONDS)) {
    $ipThrottle = ['attempts' => 0, 'first_at' => 0, 'locked_until' => 0];
}

// Session-based counters
if (!isset($_SESSION['login'])) {
    $_SESSION['login'] = ['attempts' => 0, 'first_at' => 0, 'locked_until' => 0];
}

$sessionThrottle = &$_SESSION['login'];
if ($sessionThrottle['first_at'] && ($now - $sessionThrottle['first_at'] > LOGIN_WINDOW_SECONDS)) {
    $sessionThrottle = ['attempts' => 0, 'first_at' => 0, 'locked_until' => 0];
}

// Check lockouts
$lockedSeconds = 0;
if (!empty($ipThrottle['locked_until']) && $ipThrottle['locked_until'] > $now) {
    $lockedSeconds = max($lockedSeconds, $ipThrottle['locked_until'] - $now);
}
if (!empty($sessionThrottle['locked_until']) && $sessionThrottle['locked_until'] > $now) {
    $lockedSeconds = max($lockedSeconds, $sessionThrottle['locked_until'] - $now);
}

if ($lockedSeconds > 0) {
    $error = 'Too many attempts. Please try again in ' . ceil($lockedSeconds / 60) . ' minute(s).';
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && !$lockedSeconds) {
    $email = isset($_POST['email']) ? trim((string)$_POST['email']) : '';
    $password = isset($_POST['password']) ? (string)$_POST['password'] : '';

    if ($email === '' || $password === '') {
        $error = 'Please enter your email and password.';
    } else {
        try {
            $stmt = $pdo->prepare('SELECT id, email, password_hash FROM admins WHERE email = ?');
            $stmt->execute([$email]);
            $admin = $stmt->fetch();

            $ok = false;
            if ($admin && isset($admin['password_hash'])) {
                // Use password_verify; avoid timing attacks by always calling it with a string
                $ok = password_verify($password, $admin['password_hash']);
            } else {
                // Fallback hashing to equalize timing when user not found
                password_verify($password, password_hash('dummy', PASSWORD_DEFAULT));
            }

            if ($ok) {
                // Success: reset throttles
                $ipThrottle = ['attempts' => 0, 'first_at' => 0, 'locked_until' => 0];
                save_ip_throttle($ip, $ipThrottle);
                $sessionThrottle = ['attempts' => 0, 'first_at' => 0, 'locked_until' => 0];
                $_SESSION['login'] = $sessionThrottle;

                session_regenerate_id(true);
                $_SESSION['admin_logged_in'] = true;
                $_SESSION['admin_id'] = $admin['id'];
                $_SESSION['admin_email'] = $admin['email'];
                header('Location: dashboard.php');
                exit;
            } else {
                // Failure: increment throttles
                if (!$ipThrottle['first_at']) $ipThrottle['first_at'] = $now;
                if (!$sessionThrottle['first_at']) $sessionThrottle['first_at'] = $now;
                $ipThrottle['attempts'] = (int)$ipThrottle['attempts'] + 1;
                $sessionThrottle['attempts'] = (int)$sessionThrottle['attempts'] + 1;

                if ($ipThrottle['attempts'] >= LOGIN_MAX_ATTEMPTS) {
                    $ipThrottle['locked_until'] = $now + LOGIN_LOCK_SECONDS;
                }
                if ($sessionThrottle['attempts'] >= LOGIN_MAX_ATTEMPTS) {
                    $sessionThrottle['locked_until'] = $now + LOGIN_LOCK_SECONDS;
                }
                save_ip_throttle($ip, $ipThrottle);
                $_SESSION['login'] = $sessionThrottle;

                // Small delay to slow brute force
                usleep(300000); // 300ms
                $error = 'Invalid email or password.';
            }
        } catch (Throwable $e) {
            // Avoid leaking details to user
            $error = 'An error occurred. Please try again later.';
        }
    }
}
?>
<!DOCTYPE html>
<html>
<head><title>Admin Login</title></head>
<body>
    <h2>Admin Login</h2>
    <?php if (!empty($error)): ?>
        <p style="color: red;"><?= htmlspecialchars($error) ?></p>
    <?php endif; ?>
    <form method="post" autocomplete="off">
        <label>Email:
            <input type="email" name="email" required>
        </label><br>
        <label>Password:
            <input type="password" name="password" required>
        </label><br>
        <button type="submit">Login</button>
    </form>
    <p style="margin-top:12px;"><a href="register.php">Register (invite)</a></p>
</body>
</html>
