<?php
require_once '../includes/auth.php';
require_once '../includes/csrf.php';

$ticket_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if ($ticket_id <= 0) {
    http_response_code(400);
    echo "Ticket id required";
    exit;
}

$csrfToken = csrf_token();
?>
<?php
// Compute application base by stripping the '/admin' segment from the script path so API_CRUD points to the app root.
$scriptName = $_SERVER['SCRIPT_NAME'] ?? '';
$adminPos = strpos($scriptName, '/admin/');
if ($adminPos !== false) {
  $appBase = substr($scriptName, 0, $adminPos);
} else {
  $adminPos = strpos($scriptName, '/admin');
  if ($adminPos !== false) {
    $appBase = substr($scriptName, 0, $adminPos);
  } else {
    // fallback to parent dir
    $appBase = rtrim(dirname($scriptName), '/');
  }
}
if ($appBase === '') $appBase = '';
?>
<!DOCTYPE html>
<html>
<head>
  <meta charset="utf-8">
  <title>Manage To‑Do Items for Ticket #<?= htmlspecialchars($ticket_id) ?></title>
  <meta name="csrf-token" content="<?= htmlspecialchars($csrfToken) ?>">
  <style>
    body{font-family:Arial, sans-serif;margin:18px}
    .toolbar{display:flex;gap:8px;align-items:center;margin-bottom:12px}
    #items{list-style:none;padding:0;margin:0;max-width:900px}
    #items li{border:1px solid #ddd;padding:10px;margin-bottom:6px;background:#fff;display:flex;align-items:center;gap:8px}
    .drag-handle{cursor:grab;padding:4px 8px;background:#f4f4f4;border-radius:4px}
    .completed{opacity:.6;text-decoration:line-through}
    .small{font-size:90%;color:#666}
    .actions button{margin-left:8px}
    #templateModal{position:fixed;left:0;top:0;right:0;bottom:0;background:rgba(0,0,0,.4);display:none;align-items:center;justify-content:center}
    #templateModal .panel{background:#fff;padding:16px;border-radius:6px;max-width:600px;width:100%}
  </style>
</head>
<body>
  <h1>Ticket #<?= htmlspecialchars($ticket_id) ?> — To‑Do Items</h1>
  <p><a href="dashboard.php">Back to Dashboard</a></p>

  <div class="toolbar">
    <button id="btnAdd">Add Item</button>
    <button id="btnApplyTemplate">Apply Template</button>
    <span class="small">Drag handle to reorder</span>
    <div style="margin-left:auto;font-size:90%">Average time: <span id="avgTime">—</span> min</div>
  </div>

  <div id="editor" style="display:none; margin-bottom:12px;">
    <form id="addForm" onsubmit="return onAddSubmit(event)">
      <input id="newText" placeholder="Item text" style="width:60%" required>
      <label class="small">Subitems (comma separated): <input id="newSubitems" style="width:30%"></label>
      <button type="submit">Save</button>
      <button type="button" onclick="hideEditor()">Cancel</button>
    </form>
  </div>

  <ul id="items"></ul>

  <div id="templateModal">
    <div class="panel">
      <h3>Apply Template</h3>
      <p class="small">Choose a client template to apply to this ticket. This will create items and subitems under the ticket.</p>
      <label>Client ID: <input id="tplClientId" type="number" min="1"></label>
      <button id="tplFetch">Load Templates</button>
      <div id="tplList" style="margin-top:12px"></div>
      <div style="margin-top:12px;text-align:right"><button id="tplClose">Close</button></div>
    </div>
  </div>

  <div id="toastContainer" style="position:fixed;right:16px;top:16px;z-index:9999"></div>

  <script>
  const API_CRUD = '<?= $appBase ?>/includes/crud.php';
  const CSRF = document.querySelector('meta[name="csrf-token"]').getAttribute('content');

  function showToast(msg, type){
    const c = document.getElementById('toastContainer');
    const t = document.createElement('div'); t.textContent = msg; t.className = 'toast ' + (type||'info'); t.style.background = type==='error'?'#c62828':'#333'; t.style.color='white'; t.style.padding='8px'; t.style.borderRadius='4px'; t.style.marginTop='6px';
    c.appendChild(t); setTimeout(()=>t.remove(),3000);
  }

  // Minimal apiFetch local wrapper that mirrors dashboard behavior
  async function apiFetch(path, method='GET', body=null){
    const init = { method: method, headers: {} };
    if(method !== 'GET'){
      init.headers['Content-Type'] = 'application/json';
      init.headers['X-CSRF-Token'] = CSRF;
    }
    if(body != null) init.body = JSON.stringify(body);
    const res = await fetch(path, init);
    const text = await res.text();
    let json = null; try{ json = text ? JSON.parse(text) : null } catch(e){}
    if(!res.ok){ const msg = (json && json.error) ? json.error : (text || res.statusText); showToast('API error: ' + msg, 'error'); throw new Error(msg); }
    return json;
  }

  const ticketId = <?= json_encode($ticket_id) ?>;

  async function loadItems(){
    try{
  const url = API_CRUD + '?entity=todo_items&ticket_id=' + encodeURIComponent(ticketId) + '&order_by=position';
      const json = await apiFetch(url, 'GET');
      const items = json && Array.isArray(json.data) ? json.data : [];
      // build tree
      const tree = buildTree(items);
      renderTree(tree);
      // compute average minutes across all todo items with actual_minutes
      const all = items.filter(it => it.actual_minutes != null).map(it => parseInt(it.actual_minutes,10));
      const avg = all.length ? Math.round(all.reduce((a,b)=>a+b,0)/all.length) : null;
      document.getElementById('avgTime').textContent = avg !== null ? String(avg) : '—';
    }catch(e){ console.error(e); }
  }

  function buildTree(items){
    const map = new Map();
    items.forEach(it => {
      // normalize fields
      map.set(String(it.id), Object.assign({}, it, { children: [] }));
    });
    const roots = [];
    map.forEach(node => {
      const pid = node.parent_id || node.parent || null;
      if (pid) {
        const p = map.get(String(pid));
        if (p) p.children.push(node);
        else roots.push(node); // orphan -> treat as root
      } else {
        roots.push(node);
      }
    });
    // Optionally sort children by position
    function sortRec(arr){ arr.sort((a,b)=> (a.position||0) - (b.position||0)); arr.forEach(c=> sortRec(c.children)); }
    sortRec(roots);
    return roots;
  }

  function renderTree(nodes){
    const ul = document.getElementById('items'); ul.innerHTML = '';
    nodes.forEach(n => ul.appendChild(renderNode(n, 0)));
  }

  function renderNode(node, depth){
    const li = document.createElement('li');
    li.dataset.id = node.id;
    li.style.marginLeft = (depth * 14) + 'px';
    // only top-level are draggable for simplicity
    if (!node.parent_id) li.draggable = true;
    const hasChildren = node.children && node.children.length;
    const title = node.description || node.text || '';
    li.innerHTML = '<div class="drag-handle">☰</div>' +
      '<button class="toggle" style="width:28px">' + (hasChildren ? '▾' : '') + '</button>' +
      '<input type="checkbox" class="chk" ' + (node.is_completed ? 'checked' : '') + '>' +
      '<div style="flex:1"><div class="text ' + (node.is_completed? 'completed':'') + '">' + escapeHtml(title) + '</div>' +
      '<div class="small child-count">' + (hasChildren ? (node.children.length + ' subitems') : '') + '</div></div>' +
      '<div style="display:flex;flex-direction:column;gap:6px;align-items:flex-end">' +
      '<input class="actual-min" type="number" placeholder="min" style="width:72px" value="' + (node.actual_minutes? escapeHtml(String(node.actual_minutes)) : '') + '">' +
      '<div class="actions"><button class="add-sub">Add Sub</button><button class="edit">Edit</button><button class="del">Delete</button></div>' +
      '</div>';

    // events
    const chk = li.querySelector('.chk'); chk.addEventListener('change', ()=> toggleComplete(node.id, li));
    li.querySelector('.edit').addEventListener('click', ()=> beginInlineEdit(node, li));
    li.querySelector('.del').addEventListener('click', ()=> deleteItem(node.id));
    li.querySelector('.add-sub').addEventListener('click', ()=> showAddSubForm(node, li));
    const minInput = li.querySelector('.actual-min');
    if(minInput){ minInput.addEventListener('change', async ()=>{
      const v = minInput.value.trim(); const val = v === '' ? null : parseInt(v,10);
  try{ await apiFetch(API_CRUD, 'PUT', { entity: 'todo_items', id: node.id, actual_minutes: val }); showToast('Saved','success'); await loadItems(); }catch(e){ showToast('Save failed: ' + e.message,'error'); }
    }); }
    const toggleBtn = li.querySelector('.toggle');
    if (toggleBtn) toggleBtn.addEventListener('click', ()=>{
      const childUl = li.querySelector('ul');
      if (!childUl) return; childUl.style.display = (childUl.style.display === 'none' ? '' : 'none');
      toggleBtn.textContent = (childUl.style.display === 'none' ? '▸' : '▾');
    });

    // drag handlers on draggable nodes
    if (li.draggable) {
      li.addEventListener('dragstart', onDragStart);
      li.addEventListener('dragover', onDragOver);
      li.addEventListener('drop', onDrop);
      li.addEventListener('dragend', onDragEnd);
    }

    // children
    if (hasChildren) {
      const cul = document.createElement('ul'); cul.style.listStyle='none'; cul.style.padding='0'; cul.style.margin='6px 0 0 0';
      node.children.forEach(c => cul.appendChild(renderNode(c, depth+1)));
      li.appendChild(cul);
    }

    return li;
  }

  function escapeHtml(s){ const d = document.createElement('div'); d.textContent = s==null?'':s; return d.innerHTML; }

  // CRUD operations
  async function onAddSubmit(ev){ ev.preventDefault(); const text = document.getElementById('newText').value.trim(); const subs = document.getElementById('newSubitems').value.trim(); if(!text) return showToast('Enter text','error');
    const payload = { entity: 'todo_items', ticket_id: ticketId, text: text, subitems: subs ? subs.split(',').map(s => s.trim()).filter(Boolean) : [] };
    // optimistic add: show temporary id
    const tempId = 'tmp-' + Date.now(); renderItems([{ id: tempId, text: payload.text, subitems: payload.subitems, completed: false }].concat(Array.from(document.querySelectorAll('#items li')).map(li => ({ id: li.dataset.id, text: li.querySelector('.text').textContent, completed: li.querySelector('.chk').checked, subitems: [] }))))
    hideEditor();
    try{
  await apiFetch(API_CRUD, 'POST', payload);
      showToast('Item added','success');
      await loadItems();
    }catch(e){ showToast('Failed to add: ' + e.message,'error'); await loadItems(); }
    return false;
  }

  function showEditor(){ document.getElementById('editor').style.display = ''; document.getElementById('newText').focus(); }
  function hideEditor(){ document.getElementById('editor').style.display = 'none'; document.getElementById('newText').value=''; document.getElementById('newSubitems').value=''; }

  function beginEdit(item){ showEditor(); document.getElementById('newText').value = item.text || ''; document.getElementById('newSubitems').value = (item.subitems || []).join(', ');
    // replace submit handler to do PUT
    const form = document.getElementById('addForm');
    form.onsubmit = async function(e){ e.preventDefault(); const text = document.getElementById('newText').value.trim(); const subs = document.getElementById('newSubitems').value.trim(); const payload = { entity:'todo_items', id: item.id, text: text, subitems: subs? subs.split(',').map(s=>s.trim()).filter(Boolean):[] };
  try{ await apiFetch(API_CRUD, 'PUT', payload); showToast('Item updated','success'); hideEditor(); await loadItems(); }catch(err){ showToast('Update failed: ' + err.message,'error'); }
    };
  }

  async function deleteItem(id){ if(!confirm('Delete this item?')) return; try{ await apiFetch(API_CRUD + '?entity=todo_items&id=' + encodeURIComponent(id), 'DELETE'); showToast('Deleted', 'success'); await loadItems(); }catch(e){ showToast('Delete failed: ' + e.message,'error'); }}

  async function toggleComplete(id, li){ const checked = li.querySelector('.chk').checked; try{ await apiFetch(API_CRUD, 'PUT', { entity:'todo_items', id: id, is_completed: checked ? 1 : 0 }); li.querySelector('.text').classList.toggle('completed', checked); showToast('Updated','success'); }catch(e){ showToast('Failed to update: ' + e.message,'error'); await loadItems(); }}

  function beginInlineEdit(node, li){
    // replace the title with an inline input and Save/Cancel
    const textDiv = li.querySelector('.text');
    const orig = textDiv.textContent;
    const input = document.createElement('input'); input.type='text'; input.value = orig; input.style.width='80%';
    const save = document.createElement('button'); save.textContent='Save';
    const cancel = document.createElement('button'); cancel.textContent='Cancel';
    textDiv.innerHTML=''; textDiv.appendChild(input); textDiv.appendChild(save); textDiv.appendChild(cancel);
    save.addEventListener('click', async ()=>{
      const v = input.value.trim(); if(!v) return showToast('Enter text','error');
      try{ await apiFetch(API_CRUD, 'PUT', { entity:'todo_items', id: node.id, description: v }); showToast('Saved','success'); await loadItems(); }catch(e){ showToast('Save failed: ' + e.message,'error'); }
    });
    cancel.addEventListener('click', ()=>{ textDiv.textContent = orig; });
  }

  function showAddSubForm(node, li){
    // insert a small form under this li to add a child
    if (li.querySelector('.add-sub-form')) return; // already open
    const form = document.createElement('div'); form.className = 'add-sub-form'; form.style.marginTop='8px';
    form.innerHTML = '<input class="sub-text" placeholder="Subitem text" style="width:60%" /> <button class="sub-save">Add</button> <button class="sub-cancel">Cancel</button>';
    li.appendChild(form);
    form.querySelector('.sub-save').addEventListener('click', async ()=>{
      const val = form.querySelector('.sub-text').value.trim(); if(!val) return showToast('Enter text','error');
  try{ await apiFetch(API_CRUD, 'POST', { entity:'todo_items', ticket_id: ticketId, parent_id: node.id, description: val }); showToast('Subitem added','success'); await loadItems(); }catch(e){ showToast('Failed: ' + e.message,'error'); }
    });
    form.querySelector('.sub-cancel').addEventListener('click', ()=> form.remove());
  }

  // Drag & drop reordering: on drop compute new positions and send batch update
  let dragSrc = null;
  function onDragStart(e){ this.style.opacity = '0.5'; dragSrc = this; e.dataTransfer.effectAllowed = 'move'; }
  function onDragOver(e){ e.preventDefault(); e.dataTransfer.dropEffect = 'move'; const target = this; const ul = target.parentNode; const rect = target.getBoundingClientRect(); const next = (e.clientY - rect.top) > (rect.height/2); ul.insertBefore(dragSrc, next ? target.nextSibling : target); }
  function onDrop(e){ e.stopPropagation(); }
  function onDragEnd(e){ this.style.opacity=''; // collect ordered ids
    const ids = Array.from(document.querySelectorAll('#items li')).map((li, idx) => ({ id: li.dataset.id, position: idx })).filter(Boolean);
    if (ids.length === 0) return;
    (async ()=>{
      try{
  await apiFetch(API_CRUD, 'PUT', { reorder: ids, entity: 'todo_items' });
        showToast('Reordered','success');
        await loadItems();
      }catch(e){ showToast('Reorder failed: ' + e.message,'error'); await loadItems(); }
    })();
  }

  // Apply Template flow
  document.getElementById('btnApplyTemplate').addEventListener('click', ()=>{ document.getElementById('templateModal').style.display = 'flex'; });
  document.getElementById('tplClose').addEventListener('click', ()=>{ document.getElementById('templateModal').style.display = 'none'; document.getElementById('tplList').innerHTML = ''; });
  document.getElementById('tplFetch').addEventListener('click', async ()=>{
    const clientId = document.getElementById('tplClientId').value; if(!clientId) return showToast('Enter client id','error');
    try{
  const json = await apiFetch(API_CRUD + '?entity=todo_templates&client_id=' + encodeURIComponent(clientId), 'GET');
      const list = json && Array.isArray(json.data) ? json.data : [];
      const cont = document.getElementById('tplList'); cont.innerHTML = '';
      if(list.length===0) cont.textContent = 'No templates for this client.';
      list.forEach(t => {
        const div = document.createElement('div'); div.style.border='1px solid #eee'; div.style.padding='8px'; div.style.marginBottom='6px';
        div.innerHTML = '<strong>' + escapeHtml(t.title || 'Untitled') + '</strong> <button data-id="'+t.id+'">Apply</button>' + (t.description? '<div class="small">'+escapeHtml(t.description)+'</div>':'');
        const btn = div.querySelector('button'); btn.addEventListener('click', ()=> applyTemplate(t.id));
        cont.appendChild(div);
      });
    }catch(e){ showToast('Failed to load templates: ' + e.message,'error'); }
  });

  async function applyTemplate(templateId){ if(!confirm('Apply template to this ticket?')) return; try{
    // fetch template items (server should return structure with items and subitems)
  const json = await apiFetch(API_CRUD + '?entity=todo_templates&id=' + encodeURIComponent(templateId), 'GET');
    const template = json && json.data && json.data[0] ? json.data[0] : null;
    if(!template){ showToast('Template not found','error'); return; }
    // template.items expected as array of { text, subitems: [...] }
    if(!Array.isArray(template.items)) template.items = [];
    for(const it of template.items){
      const payload = { entity:'todo_items', ticket_id: ticketId, text: it.text, subitems: it.subitems || [] };
  await apiFetch(API_CRUD, 'POST', payload);
    }
    showToast('Template applied','success');
    document.getElementById('templateModal').style.display = 'none';
    await loadItems();
  }catch(e){ showToast('Failed to apply template: ' + e.message,'error'); }}

  document.getElementById('btnAdd').addEventListener('click', ()=>{ // reset addForm handler
    const form = document.getElementById('addForm'); form.onsubmit = onAddSubmit; showEditor(); });

  // init
  loadItems();
  </script>
</body>
</html>
